// Sticky Elements specific utils, used accross files

// External dependencies
import filter from 'lodash/filter';
import forEach from 'lodash/forEach';
import get from 'lodash/get';
import head from 'lodash/head';
import includes from 'lodash/includes';
import isEmpty from 'lodash/isEmpty';
import isString from 'lodash/isString';
import $ from 'jquery';

// Internal dependencies
import {
  getOffsets,
} from './utils';

/**
 * Get top / bottom limit attributes
 *
 * @since ??
 *
 * @param {object} $selector
 * @param {string}
 *
 * @return {object}
 * @return {string} object.limit
 * @return {number} object.height
 * @return {number} object.width
 * @return {object} object.offsets
 * @return {number} object.offsets.top
 * @return {number} object.offsets.right
 * @return {number} object.offsets.bottom
 * @return {number} object.offsets.left
 */
export const getLimit = ($selector, limit) => {
  // @todo update valid limits based on selector
  const validLimits = ['body', 'section', 'row', 'column'];

  if (!includes(validLimits, limit)) {
    return false;
  }

  // Limit selector
  const $limitSelector = getLimitSelector($selector, limit);

  if (!$limitSelector) {
    return false;
  }

  const height = $limitSelector.outerHeight();
  const width = $limitSelector.outerWidth();

  return {
    limit,
    height,
    width,
    offsets: getOffsets($limitSelector, width, height),
  };
}

/**
 * Get top / bottom limit selector based on given name
 *
 * @since ??
 *
 * @param {object} $selector
 * @param {string} limit
 *
 * @return {bool|object}
 */
export const getLimitSelector = ($selector, limit) => {
  let parentSelector = false;

  switch (limit) {
    case 'body':
      parentSelector = '.et_builder_inner_content';
      break;
    case 'section':
      parentSelector = '.et_pb_section';
      break;
    case 'row':
      parentSelector = '.et_pb_row';
      break;
    case 'column':
      parentSelector = '.et_pb_column';
      break;
    default:
      break;
  }

  return parentSelector ? $selector.closest(parentSelector) : false;
}

/**
 * Filter invalid sticky modules
 * 1. Sticky module inside another sticky module
 *
 * @param {object} modules
 * @param {object} currentModules
 *
 * @since ??
 */
export const filterInvalidModules = (modules, currentModules = {}) => {
  const filteredModules = {};

  forEach(modules, (module, key) => {
    // If current sticky module is inside another sticky module, ignore current module
    if ($(module.selector).parents('.et_pb_sticky_module').length > 0) {
      return;
    }

    // Repopulate the module list
    if (!isEmpty(currentModules) && currentModules[key]) {
      // Keep props that isn't available on incoming modules intact
      filteredModules[key] = {
        ...currentModules[key],
        ...module,
      };
    } else {
      filteredModules[key] = module;
    }
  });

  return filteredModules;
}

/**
 * Get sticky style of given module by cloning, adding sticky state classname, appending DOM,
 * retrieving value, then immediately the cloned DOM. This is needed for property that is most
 * likely to be affected by transition if the sticky value is retrieved on the fly, thus it needs
 * to be retrieved ahead its time by this approach
 *
 * @since ??
 *
 * @param {object} $module
 * @param {string} id
 *
 * @return {object}
 */
export const getStickyStyles = (id, $module) => {
  // Sticky state classname to be added; these will make cloned module to have fixed position and
  // make sticky style take effect
  const stickyStyleClassname = 'et_pb_sticky et_pb_sticky_style_dom';

  // Cloned the module add sticky state classname; set the opacity to 0 and remove the transition
  // so the dimension can be immediately retrieved
  const $stickyStyleDom = $module.clone().addClass(stickyStyleClassname).attr({
    'data-sticky-style-dom-id': id,
  }).css({
    opacity: 0,
    transition: 'none',
    animation: 'none',
  });

  // Cloned module might contain image. However the image might take more than a milisecond to be
  // loaded on the cloned module after the module is appended to the layout EVEN IF the image on
  // the $module has been loaded. This might load to inaccurate sticky style calculation. To avoid
  // it, recreate the image by getting actual width and height then recreate the image using SVG
  $stickyStyleDom.find('img').each(function (index) {
    const $img = $(this);
    const $measuredImg = $module.find(`img:eq(${index})`);
    const measuredWidth = get($measuredImg, [0, 'naturalWidth'], $module.find(`img:eq(${index})`).outerWidth());
    const measuredHeight = get($measuredImg, [0, 'naturalHeight'], $module.find(`img:eq(${index})`).outerHeight());

    $img.attr({
      // Remove scrse to force DOM to use src
      scrset: '',

      // Recreate svg to use image's actual width so the image reacts appropriately when sticky
      // style modifies image dimension (eg image has 100% and padding in sticky style is larger;
      // this will resulting in image being smaller because the wrapper dimension is smaller)
      src: `data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" width="${measuredWidth}" height="${measuredHeight}"><rect width="${measuredWidth}" height="${measuredHeight}" /></svg>`
    });
  });

  // Append the cloned DOM
  $module.after($stickyStyleDom);

  // Measure sticky style DOM properties
  const styles = {
    height: $stickyStyleDom.outerHeight(),
    width: $stickyStyleDom.outerWidth(),
  };

  // Immediately remove the cloned DOM
  $(`.et_pb_sticky_style_dom[data-sticky-style-dom-id="${id}"]`).remove();

  return styles;
}

/**
 * Remove given property's transition from transition property's value. To make some properties
 * (eg. width, top, left) transition smoothly when entering / leaving sticky state, its property
 * and transition need to be removed then re-added 50ms later. This is mostly happened because the
 * module positioning changed from relative to fixed when entering/leaving sticky state
 *
 * @since ??
 *
 * @param {string} transitionValue
 * @param {array} trimmedProperties
 *
 * @return {string}
 */
export const trimTransitionValue = (transitionValue, trimmedProperties) => {
  // Make sure that transitionValue is string. Otherwise split will throw error
  if (!isString(transitionValue)) {
    transitionValue = '';
  }

  const transitions  = transitionValue.split(', ');
  const trimmedValue = filter(transitions, (transition) => {
    return !includes(trimmedProperties, head(transition.split(' ')));
  });

  return isEmpty(trimmedValue) ? 'none' : trimmedValue.join(', ');
}

/**
 * Calculate automatic offset that should be given based on sum of heights of all sticky modules
 * that are currently in sticky state when window reaches $target's offset.
 *
 * @since ??
 *
 * @param {object} $target
 *
 * @return {number}
 */
export const getClosestStickyModuleOffsetTop = ($target) => {
  const offset = $target.offset();
  offset.right = offset.left + $target.outerWidth();

  let closestStickyElement   = null;
  let closestStickyOffsetTop = 0;

  // Get all sticky module data from store. NOTE: this util might be used on various output build
  // so it needs to get sticky store value via global object instead of importing it
  const stickyModules = get(window.ET_FE, 'stores.sticky.modules', {});

  // Loop sticky module data to get the closest sticky module to given y offset. Sticky module
  // already has map of valid modules it needs to consider as automatic offset due to
  // adjacent-column situation.
  // @see https://github.com/elegantthemes/Divi/issues/19432
  forEach(stickyModules, stickyModule => {
    // Ignore sticky module if it is stuck to bottom
    if (!includes(['top_bottom', 'top'], stickyModule.position)) {
      return;
    }

    // Ignore if sticky module's right edge doesn't collide with target's left edge
    if (get(stickyModule, 'offsets.right', 0) < offset.left) {
      return;
    }

    // Ignore if sticky module's left edge doesn't collide with target's right edge
    if (get(stickyModule, 'offsets.left', 0) > offset.right) {
      return;
    }

    // Ignore sticky module if it is located below given y offset
    if (get(stickyModule, 'offsets.top', 0) > offset.top) {
      return;
    }

    // Ignore sticky module if its bottom limit is higher than given y offset
    const bottomLimitBottom = get(stickyModule, 'bottomLimitSettings.offsets.bottom');

    if (bottomLimitBottom && bottomLimitBottom < offset.top) {
      return;
    }

    closestStickyElement = stickyModule;
  });

  // Once closest sticky module to given y offset has been found, loop its topOffsetModules, get
  // each module's heightSticky and return the sum of their heights
  if (get(closestStickyElement, 'topOffsetModules', false)) {
    forEach(get(closestStickyElement, 'topOffsetModules', []), stickyId => {
      // Get sticky module's height on sticky state; fallback to height just to be safe
      const stickyModuleHeight = get(
        stickyModules,
        [stickyId, 'heightSticky'],
        get(stickyModules, [stickyId, 'height'], 0)
      );

      // Sum up top offset module's height
      closestStickyOffsetTop += stickyModuleHeight;
    });

    // Get closest-to-y-offset's sticky module's height on sticky state;
    const closestStickyElementHeight = get(
      stickyModules,
      [closestStickyElement.id, 'heightSticky'],
      get(stickyModules, [closestStickyElement.id, 'height'], 0)
    );

    // Sum up top offset module's height
    closestStickyOffsetTop += closestStickyElementHeight;
  }

  return closestStickyOffsetTop;
}
